/*
 * 	termometer.c ... firmware for open hardware wireless termometer
 *
 * 	Copyright (C) 2007-2014 Jiri Pittner <jiri@pittnerovi.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 	                                         
 * 	                                        
*/

#undef debug

//atmega8, 14.7456
// make fuse_atmega_sut11
//upload also eeprom!
//
#define LOCATION generic
#define COUNTER_START 0
#define SENSOR_SERIAL 1


/*PIN assignment

433 modules:
	PB1 OUT tx433
	PB2 OUT tx_on

LCD display:
	PC0-3=DB4-7
	PC4=E
	PC5=RW
	PD4=RS or PD4  on atmega8

temperature:
	PB0 (SMT160-30)

*/ 

//corresponds to schematics
#define TEMPPORT PINB
#define TEMPPIN PB0

#define MEASUREMODULO 0x0f
#define AVERAGING 8
#define BAUD 115200
#define TXCOUNT 4

#include "backward.h"
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <setjmp.h>
#include <avr/sleep.h>
//obsolete #include <avr/timer.h>
#include <avr/wdt.h>
#include <stdlib.h>
#include <string.h>

#define INCFILE_PARAMS_(X) X##_params.h
#define INCFILE_PARAMS(X) INCFILE_PARAMS_(X)
#define INCFILE_KEYS_(X) X##_keys.h
#define INCFILE_KEYS(X) INCFILE_KEYS_(X)
#define XSTR(s) STR(s)
#define STR(s) #s

#define STRLOCATION STR(LOCATION)

#include XSTR(INCFILE_PARAMS(LOCATION))
#include "keeloq2.h"
#include XSTR(INCFILE_KEYS(LOCATION))
#include "devices433.h"




#if defined(at90s2313) || defined(at90s8535)
#else
#define ATmega
#endif

#ifdef ATmega
#define USR UCSRA
#endif


#if (XTAL == 14745600L) 
#define oversampling 4
#else
#define oversampling 2
#endif

#ifndef MCU
#define emulate
#else
#define AVR_assembler
#endif

#define itoa10(N,S) itoa(N,S,10)
#define itoa16(N,S) itoa(N,S,16)


void printP (PGM_P string){
        char c;
        c=pgm_read_byte(string);
                   while (c) {
                   loop_until_bit_is_set(USR, UDRE);
                   UDR = c;
                   c=pgm_read_byte(++string);
                   }
                   return;
                 }



void print (char *string){                                     
                   while (*string) {                                            
                   loop_until_bit_is_set(USR, UDRE);                            
                   UDR = *string++;
                   }                                                            
                   return;                                                      
                 }                                                              


void scan(char *string){
char c;
	do 	{
		do {
			loop_until_bit_is_set(USR, RXC);	
			c =UDR;
			} while bit_is_set(USR, FE);	
		*string++ = c;
		//echo the character
		loop_until_bit_is_set(USR, UDRE);
		UDR = c;
		} while ( c != '\r' );
	loop_until_bit_is_set(USR, UDRE);
        UDR = '\n';
	string[-1]=0;
	}



//UART initialize
#ifdef ATmega
#define UCR UCSRB
#define UART_INIT(baud) { \
UBRRH=0; \
UBRRL= (XTAL/baud+15)/16-1; \
UCSRB=(1<<TXEN)|(1<<RXEN); \
UCSRC=(1<<URSEL)|(1<<UCSZ0)|(1<<UCSZ1)|(1<<USBS); }
#else
#define UART_INIT(baud) { \
UBRR = (XTAL/baud+15)/16-1; \
sbi(UCR, TXEN); \
sbi(UCR, RXEN); \
}
#endif


watchdog(uint8_t onoff)
{
if(onoff) {wdt_enable(WDTO_2S); wdt_reset();}
else {wdt_reset();wdt_disable();}
}


#include "keeloq2.c"

//transmit routines

volatile uint8_t outcode[MESSAGE_BYTES];
volatile uint8_t outcodelen=0; //0 indicates ready to receive, non-zero indicates ready to read it in main
volatile uint8_t waittime=0;
volatile uint8_t txcount=0; //repeat several times
volatile static uint8_t obitcount=0, obytecount=0; //working counters

typedef enum {init,oupmark,downmark,data,trailer,wait} OSTATE;
static OSTATE ostate=init;
static uint8_t ocount=0;




void txcar(void)
{
switch (ostate) {
	case wait: if(--waittime) return; 
        case init: //the user has just written the request to transmit
			//check maxcodelen
			ocount=13;
			ostate=oupmark;
			cbi(TIMSK,TOIE1);
			ICR1=(uint16_t)(XTAL*8./10000+.5)-1;
                	OCR1A=((uint16_t)(XTAL*4./10000+.5)-1);
			TCNT1=0;
			cbi(TCCR1A,COM1A0);
			cbi(TCCR1A,COM1A1);
			sbi(TIMSK,TOIE1);
			sbi(TIMSK,OCIE1A);
                break;
        case oupmark:
#if (MCU == atmega8 )
		if(--ocount) {sbi(PORTB,PB1); return;}
		cbi(PORTB,PB1);
#else //ATmega 16 and 32
		if(--ocount) {sbi(PORTD,PD5); return;}
		cbi(PORTD,PD5);
#endif
		cbi(TIMSK,TOIE1);
                TCNT1=0;
		ICR1=(uint16_t)(XTAL*6.1/10000+.5)-1;
                sbi(TIMSK,TOIE1);
		ostate=downmark;
		ocount=2;
                break;
        case downmark:
		if(--ocount) return;
		obitcount=obytecount=0;
		ostate=data;
		cbi(TIMSK,TOIE1);
		ICR1=(uint16_t)(XTAL*12./10000+.5)-1;
                sbi(TIMSK,TOIE1);
		break;
        case data:
		{
		uint8_t n;
		n=obitcount+(obytecount<<3);
		if(n>outcodelen+1)
                        {
                        //data finished 
#if (MCU == atmega8 )
                cbi(PORTB,PB1);
#else //ATmega 16 and 32
                cbi(PORTD,PD5);
#endif
			cbi(TIMSK,OCIE1A);
			ostate=trailer;
			ocount=13;
			OCR1A=0;
                        }
		else
			{
#if (MCU == atmega8 )
                        if(n) sbi(PORTB,PB1); //avoid spurious first peak
#else //ATmega 16 and 32
                        if(n) sbi(PORTD,PD5); //avoid spurious first peak
#endif
			if(n>outcodelen) 
				{
				OCR1A=((uint16_t)(XTAL*12./10000+.5)-1)*1/3;
				++obitcount;
				}
			else
				{
				OCR1A=((uint16_t)(XTAL*12./10000+.5)-1)*
				((outcode[obytecount]& (1<<obitcount))?1:2)/3;
				if(++obitcount==8) {obitcount=0;++obytecount;}
				}
			}
		}
                break;
        case trailer:
		if(--ocount) return;
		//finished
                if(--txcount == 0) {ostate=init; outcodelen=0;}
                else {ostate=wait; waittime=32;} 
                break;
}
}



INTERRUPT(SIG_OVERFLOW1)
{
if(!outcodelen) return;
txcar();
}


INTERRUPT(SIG_OUTPUT_COMPARE1A)
{
#if (MCU == atmega8 ) 
cbi(PORTB,PB1);
#else //ATmega 16 and 32
cbi(PORTD,PD5); 
#endif
}

static uint16_t counter_eeprom_output __attribute__ ((section(".eeprom"))) = COUNTER_START;

//delay routines

void delay_xs(uint16_t xs) __attribute__((noinline));
void delay_xs(uint16_t xs) //xs takes 4 clocks time
{
        asm volatile (
        "\n"
        "L_dl1%=:" "\n\t"
        "sbiw %0, 1" "\n\t"
        "brne L_dl1%=" "\n\t"
        : "=&w" (xs)
        );
        return;
}

void delay_ms(uint16_t ms) //ms takes 1/1000 s
{
while(ms--) delay_xs(XTAL/4/1000);
}


//display routines

#define lcd_delay 50

void lcd_w4bit(uint8_t rs, uint8_t x)
{
PORTC= 1<<PC4
#if (MCU == atmega8 )
	; if(rs) sbi(PORTD,PD4); else cbi(PORTD,PD4);
#else
	 | rs <<PC6;
#endif
delay_xs(lcd_delay);
PORTC |= x&0x0f;
delay_xs(lcd_delay);
cbi(PORTC,PC4);
delay_xs(lcd_delay);
sbi(PORTC,PC4);
delay_xs(lcd_delay);
}

uint8_t lcd_r4bit(uint8_t rs)
{
uint8_t r;
PORTC= 1<<PC4 | 1<<PC5
#if (MCU == atmega8 )
        ; if(rs) sbi(PORTD,PD4); else cbi(PORTD,PD4);
#else
         | rs <<PC6;
#endif
delay_xs(lcd_delay);
cbi(PORTC,PC4);
delay_xs(lcd_delay);
r=PINC&0x0f;
cbi(PORTC,PC5);
delay_xs(lcd_delay);
return r;
}

uint8_t lcd_rbyte(uint8_t rs)
{
#if (MCU == atmega8 )
sbi(DDRD,PD4);
DDRC=0x30;
#else
DDRC=0xf0;
#endif
delay_xs(lcd_delay);
return (lcd_r4bit(rs)<<4)|lcd_r4bit(rs);
#if (MCU == atmega8 )
sbi(DDRD,PD4);
DDRC=0x3f;
#else
DDRC=0xff;
#endif
delay_xs(lcd_delay);
}

void lcd_init4bit(void)
{
#if (MCU == atmega8 )
sbi(DDRD,PD4);
DDRC=0x3f;
#else
DDRC=0xff;
#endif
delay_xs(20000);
lcd_w4bit(0,3);
delay_xs(10000);
lcd_w4bit(0,3);
delay_xs(500);
lcd_w4bit(0,3);
lcd_w4bit(0,2);
}

void lcd_wbyte(uint8_t rs, uint8_t x)
{
lcd_w4bit(rs,x>>4);
lcd_w4bit(rs,x);
}

void lcd_clear(void)
{
lcd_wbyte(0,0x01);
delay_xs(30000);
}

void lcd_init(void)
{
lcd_init4bit();
lcd_wbyte(0,0x28);
lcd_wbyte(0,0x08);
lcd_clear();
lcd_wbyte(0,0x06);
lcd_wbyte(0,0x0c);
}

void lcd_print(char *t)
{
while(*t) if(*t=='\n') {++t; lcd_wbyte(0,0xc0);} else lcd_wbyte(1,*t++);
}


void lcd_print8(char *t)
{
uint8_t l=0;
while(*t) {lcd_wbyte(1,*t++); ++l;}
while(l<8) {lcd_wbyte(1,' '); ++l;}
}

void lcd_cursor(uint8_t r, uint8_t c)
{
lcd_wbyte(0,0x80+c+(r<<6));
}


//temperature routines

int16_t temperature(void) // thermometer returns in 0.1 C
{
int16_t t;
uint8_t i;
uint16_t lcount,hcount;

watchdog(1);

cbi(SREG,7); //switch off interrupts

//SMT160-30 duty=0.320+0.00470*T, T=(duty-0.32)/0.00470

lcount=hcount=0;


{
uint8_t i;
for(i=
#if (XTAL == 3686400L)
0
#else
#if (XTAL == 14745600L)
64
#else
128
#endif
#endif
; --i;)
{

//optimized compilation results in irreproducible and wrong timings, therefore inline assembly

asm volatile (

//loop_until_bit_is_set(TEMPPORT,TEMPPIN);
"TEMP1:\n\t"
"sbis %4,%5" "\n\t"
"rjmp TEMP1" "\n\t"

//loop_until_bit_is_clear(TEMPPORT,TEMPPIN);
"TEMP2:\n\t"
"sbic %4,%5" "\n\t"
"rjmp TEMP2" "\n\t"

//while(bit_is_clear(TEMPPORT,TEMPPIN)) ++lcount;
"TEMP3:\n\t"
"adiw %1,1" "\n\t"
"sbis %4,%5" "\n\t"
"rjmp TEMP3" "\n\t"

//while(bit_is_set(TEMPPORT,TEMPPIN)) ++hcount;
"TEMP4:\n\t"
"adiw %0,1" "\n\t"
"sbic %4,%5" "\n\t"
"rjmp TEMP4" "\n\t"

//parameter lists
: "=w" (hcount), "=w" (lcount) //outputs
: "0" (hcount), "1" (lcount), "M" ((uint8_t) &(TEMPPORT) - 0x20), "I" (TEMPPIN) //inputs
: "cc"  //clobber
);

}
}

#ifdef debug
{
uint32_t x;
x=10000; x*=hcount;
x /=((uint32_t)hcount+lcount);
char c[8];
itoa10(x,c);
printP(PSTR("10000*Duty=")); print(c); printP(PSTR("\n"));
}
#endif

sbi(SREG,7);

{
uint32_t x;
x=68085L; x*=hcount;
x /=((uint32_t)hcount+lcount);
t = x - 21787;
t>>= 5;
}
return t;
}




void init_random_byte(void)
{
//start timer0
TCNT0=temperature();
TCCR0=1;

}

uint8_t random_byte(void)
{
//read value of timer0
return TCNT0;
}


void do_transmit(uint8_t onoff)
{
if(onoff)
        {
#if (MCU == atmega8 )
                sbi(DDRB,PB1); cbi(PORTB,PB1);
		sbi(DDRB,PB2); sbi(PORTB,PB2);
#else //ATmega 16 and 32
		sbi(DDRD,PD7); sbi(PORTD,PD7); 
                cbi(PORTD,PD5); sbi(DDRD,PD5);
#endif
	delay_ms(1);
	TCCR1A= (1<<WGM11);
 	TCCR1B= (1<<WGM12)|(1<<WGM13)|CK;
	ICR1=(uint16_t)(XTAL*12./10000+.5)-1;
	OCR1A=0;
	TCNT1=0;
	sbi(TIMSK,TOIE1);
	}
else
	{
	TCCR1A=0;
        TCCR1B=0;
        cbi(TIMSK,TOIE1);
        cbi(DDRD,PD7); cbi(PORTD,PD7);
#if (MCU == atmega8 )
                cbi(PORTB,PB2); sbi(DDRB,PB2);
                cbi(PORTB,PB1); sbi(DDRB,PB1);
#else //ATmega 16 and 32
                cbi(PORTD,PD5); sbi(DDRD,PD5);
                cbi(PORTD,PD7); sbi(DDRD,PD7);
#endif
	}
}




//statics for output keeloq processing
static KEELOQ_CODE2 keeloq_crypt_output= LOCATION_KEELOQ_SENSOR(SENSOR_SERIAL);


#define min(x,y) ((x)<(y)?(x):(y))


static uint16_t counter_output;

void transmit(uint32_t code, uint8_t count)
{

if(!count) return;

//wait for end of previous transmit or switch on transmitter
if(txcount || outcodelen)
        {
		do {watchdog(1);delay_ms(10);} while(txcount || outcodelen);
        }
else do_transmit(1);

++counter_output;

//assemble message
message msg;
assemble_message(SENSOR_DEVICE,SENSOR_SERIAL,counter_output,code,random_byte(),&msg);

watchdog(1);

//encrypt message, result in outcode
encrypt_message(&msg, outcode, keeloq_crypt_output);

//transmit the code a few times
txcount=count;
outcodelen=MESSAGE_BYTES*8;

//do not wait for end, loop in main checks and starts receiving when finished
}


void printtemp(int16_t temp, char *t)
{
uint8_t sign = (temp<0);
if(sign) temp= -temp;
strcpy(t,"   . C");
t[4] = '0' + temp%10;
temp /= 10;
t[2] = '0' + temp%10;
temp /= 10;
if(temp)
	{
	t[1] = '0' + temp%10;
	if(sign) t[0]='-';
	}
else
	{
	if(sign) t[1]='-';
	}
}




int main(void)
{
watchdog(1);
init_random_byte();

counter_output=eeprom_read_word(&counter_eeprom_output)+256;

UART_INIT(BAUD);

//global interrupt activate
sbi(SREG,7);

//write initial message to the display and blink leds
printP(PSTR("Reset!\n"));
lcd_init();
lcd_print("Reset!");
watchdog(1);


//transmit reset report
        {
        CODE433 c;
        c.cmd.instr= THERMOMETER|RESET_REPORT;
        transmit(c.code,TXCOUNT);
        }

do{
watchdog(1);
delay_ms(100);
} while(outcodelen||txcount);
watchdog(1);
delay_ms(500);


int16_t temp,tempold,tempold2;
temp=temperature();
int16_t tempstor[AVERAGING];
uint8_t tempindex;
for(tempindex=0;tempindex<AVERAGING; ++tempindex) tempstor[tempindex]=temp;
tempold2=tempold= -10000;
tempindex=0;
uint16_t pass=0xffff;

//LOOP
while(1)
{
++pass;
if( !txcount && !outcodelen)
        {
        delay_ms(5);
        do_transmit(0);
        }

delay_ms(100);
char text[10];

watchdog(1);
if((pass& MEASUREMODULO)==0 && txcount==0 && outcodelen==0 ) //check temperature every 7-th step, but not when transmitting
	{
	tempstor[tempindex]=temperature(); //check temperature not each step and average anyway
	tempindex = (tempindex+1)%AVERAGING;
	int16_t tmp=0;
	{uint8_t i;
	for(i=0; i<AVERAGING; ++i) tmp +=tempstor[i];
	}
	temp=tmp/AVERAGING;
	}

text[0]='T';
text[1]='=';
printtemp(temp,text+2);
text[8]=0;
lcd_cursor(0,0);
lcd_print8(text);

if(temp!=tempold2) {print(text);print("\n"); tempold2=temp;}


if((pass&0x01ff)==0 && temp!=tempold || (pass&0x0fff)==0) //report temperature changes but not more frequently than approx once a minute; time to time report even if temperature remains constant
	{
	tempold=temp;
	CODE433 c;
	c.cmd.instr= THERMOMETER|TEMPERATURE_REPORT;
	c.cmd.arg1=0;
	c.cmd.arg2=temp;
	transmit(c.code,TXCOUNT);
	}

//store new counter every while to eeprom
if(counter_output>eeprom_read_word(&counter_eeprom_output)+200) eeprom_write_word(&counter_eeprom_output,counter_output);

}//while

}

