/*
 * 	Copyright (C) 2006-2014 Jiri Pittner <jiri@pittnerovi.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 	                                         
 * 	                                        
*/

#ifndef _keeloq2_c
#define _keeloq2_c

#include "keeloq2.h"

#include "keeloq.c"

uint32_t atoi10_32(char *p)
{
uint32_t n=0;
while(p && isdigit(*p))
	{
	n*=10;
	n+= *p++ - '0';
	}
return n;
}

char *itoa10_32(uint32_t n,char *p)
{
char *p0=p;
char *q=p;
do {
	*p++ = (n%10) + '0';
	n/=10;
   } while (n);
*p=0;
//reverse order of the string
char z;
--p;
while(p>q) {z=*p; *p=*q; *q=z; --p; ++q;}
return p0;
}

char *itoa16_32(uint32_t n,char *p)
{
char *p0=p;
char *q=p;
do {
	uint8_t x=n&0x0f;
	*p++ = x<10? x + '0': x-10+'a';
	n>>=4;
   } while (n);
*p=0;
//reverse order of the string
char z;
--p;
while(p>q) {z=*p; *p=*q; *q=z; --p; ++q;}
return p0;
}


void encrypt2(register uint32_t *code, const KEELOQ_CODE *key)
{
encrypt(code,key);
encrypt(code,key+1);
}

void decrypt2(register uint32_t *code, const KEELOQ_CODE *key)
{
decrypt(code,key+1);
decrypt(code,key);
}

void myhtonl(const uint32_t *i, uint8_t *p)
{
p[0]= (*i)&0xff;
p[1]=((*i)>>8)&0xff;
p[2]=((*i)>>16)&0xff;
p[3]=((*i)>>24)&0xff;
}

void myntohl(uint32_t *i, const uint8_t *p)
{
(*i)=p[3];
(*i)<<=8;
(*i)|=p[2];
(*i)<<=8;
(*i)|=p[1];
(*i)<<=8;
(*i)|=p[0];
}


void encrypt_message(message *m, portable_message em, const KEELOQ_CODE2 key)
{
uint8_t i;

//set duplicate of type and serial
m->crypt[2] &= 0xffff;
m->crypt[2] |= (uint32_t) m->data.device << 16;
m->crypt[2] |= (uint32_t) m->data.serial <<24;

//seed randomness (for AVR we have to cast)
m->crypt[0] ^= (((uint32_t)m->data.random) <<8) | m->data.random;


//cbc-crypt
for(i=0; i<3; ++i)
	{
	encrypt2(&m->crypt[i],key);
	if(i<2) m->crypt[i+1] ^= m->crypt[i];
	}

//make portable message
myhtonl(&m->crypt[0],em);
myhtonl(&m->crypt[1],em+4);
myhtonl(&m->crypt[2],em+8);
em[12]=m->data.random;
em[13]=m->data.device;
em[14]=m->data.serial;

//scrabmle
em[13] ^= em[12] ^ em[2] ^ em[5];
em[14] ^= em[12] ^ em[3] ^ em[6];

//generate parity
uint8_t parity=0;
for(i=0; i<MESSAGE_BYTES-1; ++i) parity ^= em[i];
em[MESSAGE_BYTES-1]=parity;
}


uint8_t decrypt_message(message *m, portable_message em, uint8_t len, const KEELOQ_CODE2 **key)
{
uint8_t i;

//check length
if(len!=MESSAGE_BYTES*8) return 1;

//check parity
uint8_t parity=em[MESSAGE_BYTES-1];
for(i=0; i<MESSAGE_BYTES-1; ++i) parity ^= em[i];
if(parity) return 2;

//unscramble
em[13] ^= em[12] ^ em[2] ^ em[5];
em[14] ^= em[12] ^ em[3] ^ em[6];

//make message at host endianity from portable one
myntohl(&m->crypt[0],em);
myntohl(&m->crypt[1],em+4);
myntohl(&m->crypt[2],em+8);
m->data.random=em[12];
m->data.device=em[13];
m->data.serial=em[14];

//check that device is legal - errors beyond a single parity detectable error might have happened
if(m->data.device >= MAX_DEVICE_TYPES) return 4;
if(!key[m->data.device]) return 5;

//check that serial is legal - list of defined serials is null-terminated
for(i=0; i<=m->data.serial; ++i) if(!key[m->data.device][m->data.serial] || !key[m->data.device][m->data.serial][0]) return 3;

//cbc-decrypt
for(i=3; i!=0; --i)
        {
        decrypt2(&m->crypt[i-1],key[m->data.device][m->data.serial]);
        if(i>1) m->crypt[i-1] ^= m->crypt[i-2];
        }


//remove randomness - for AVR we have to cast
m->crypt[0] ^=  (((uint32_t)m->data.random) <<8) | m->data.random;

return 0;
}

#ifndef NO_COUNTER16
void assemble_message(DEVICE device, SERIAL serial, const uint16_t counter, uint32_t payload, uint8_t rnd, message *m)
{

m->data.random=rnd;
m->data.device=device;
m->data.serial=serial;

m->crypt[0] = counter;
m->crypt[0] ^= ((uint32_t)rnd)| (((uint32_t)rnd)<<8);
m->crypt[0] &= 0xffffL; //some garbage appears on AVR otherwise (due to wrong extension of rnd)
m->crypt[0] |= (payload&0xffff)<<16;

m->crypt[1] = (counter*3)&0xffff;
m->crypt[1] |= (payload&0xffff0000L);

m->crypt[2] = (counter*7)&0xffff;
m->crypt[2] |= ((uint32_t)device)<<16;
m->crypt[2] |= ((uint32_t)serial)<<24;

}

uint8_t disassemble_message(DEVICE *device, SERIAL *serial, uint16_t *counter, uint32_t *payload, const message *m)
{
uint16_t test;
uint16_t rnd=m->data.random;


//check duplicate of type and serial
if( (( m->crypt[2] >>16 ) &0xff) != m->data.device) return 10;
if( (( m->crypt[2] >>24 ) &0xff) != m->data.serial) return 20;
*device = m->data.device;
*serial = m->data.serial;

*counter = (m->crypt[0]&0xffff) ^ (rnd|(rnd<<8));
*payload = (m->crypt[0] >>16) & 0xffff;
*payload |= (m->crypt[1]&0xffff0000L);

test=*counter * 7; if(test != (m->crypt[2]&0xffff)) return 40;
test=*counter * 3; if(test != (m->crypt[1]&0xffff)) return 30;

return 0;
}

uint8_t validate_message(DEVICE devicetype, SERIAL nserial, uint16_t *counters, uint16_t *resync, DEVICE device, SERIAL serial, uint16_t counter)
{
if(device!=devicetype) return 1;
if(serial>=nserial) return 2;

//handle resync now
if(resync[serial])
        {
        if(counter==resync[serial]) return 4; //accidentally repeatedly received same, stay in resync mode
        if(counter==resync[serial]+1) //success
                {
#ifdef debug
                printP(PSTR("resync OK\n"));
#endif
                counters[serial]=counter;
                resync[serial]=0;
		return 0;
                }
        else
                {
#ifdef debug
                printP(PSTR("resync failed\n"));
#endif
                resync[serial]=0;
                return 3;
                }
        }


//ignore if counter is equal or below stored counter
//request resync if counter is more than 16 advanced

{
int16_t cdiff = counter - counters[serial];
if(cdiff<=0) return 5;
if(cdiff>16) //needs resync
        {
#ifdef debug
        printP(PSTR("resync requested\n"));
#endif
        resync[serial]=counter;
        return 6;
        }
}
counters[serial]=counter; //store recent value
return 0; //success
}

#endif
//NO_COUNTER16


#ifdef COUNTER32
void assemble_message32(DEVICE device, SERIAL serial, uint32_t counter, uint32_t payload, uint8_t rnd, message *m)
{
uint16_t counterl = counter & 0xffff;
uint16_t counterh = counter >>16;

m->data.random=rnd;
m->data.device=device;
m->data.serial=serial;

m->crypt[0] = counterl;
m->crypt[0] ^= ((uint32_t)rnd)| (((uint32_t)rnd)<<8);
m->crypt[0] &= 0xffffL; //some garbage appears on AVR otherwise (due to wrong extension of rnd)
m->crypt[0] |= (payload&0xffff)<<16;

m->crypt[1] = ((counterl*3)&0xffff) ^ counterh;
m->crypt[1] |= (payload&0xffff0000L);

m->crypt[2] = (counterl*7)&0xffff ^ ((counterh*3)&0xffff);
m->crypt[2] |= ((uint32_t)device)<<16;
m->crypt[2] |= ((uint32_t)serial)<<24;

}

uint8_t disassemble_message32(DEVICE *device, SERIAL *serial, uint32_t *counter, uint32_t *payload, const message *m)
{
uint16_t test;
uint16_t rnd=m->data.random;

uint16_t counterl,counterh;

//check duplicate of type and serial
if( (( m->crypt[2] >>16 ) &0xff) != m->data.device) return 10;
if( (( m->crypt[2] >>24 ) &0xff) != m->data.serial) return 20;
*device = m->data.device;
*serial = m->data.serial;

counterl = (m->crypt[0]&0xffff) ^ (rnd|(rnd<<8));
*payload = (m->crypt[0] >>16) & 0xffff;
*payload |= (m->crypt[1]&0xffff0000L);

counterh = m->crypt[1] ^ ((counterl*3)&0xffff);
test= (counterl * 7) ^ (counterh * 3); if(test != (m->crypt[2]&0xffff)) return 40;

*counter = counterh;
*counter <<= 16;
*counter |= counterl;

return 0;
}


uint8_t validate_message32(DEVICE devicetype, SERIAL nserial, uint32_t *counters, uint32_t *resync, DEVICE device, SERIAL serial, uint32_t counter, uint8_t longcounter)
{
if(device!=devicetype) return 1;
if(serial>=nserial) return 2;

//handle resync now
if(resync[serial])
        {
        if(counter==resync[serial]) return 4; //accidentally repeatedly received same, stay in resync mode
        if(longcounter? (counter==resync[serial]+1) :((counter&0xffff) == ((resync[serial]+1)&0xffff)) ) //success
                {
#ifdef debug
                printP(PSTR("resync OK\n"));
#endif
                counters[serial]=counter;
                resync[serial]=0;
		return 0;
                }
        else
                {
#ifdef debug
                printP(PSTR("resync failed\n"));
#endif
                resync[serial]=0;
                return 3;
                }
        }


//ignore if counter is equal or below stored counter
//request resync if counter is more than 16 advanced
//ignore possibility of overflow here

if(!longcounter)
{
int16_t cdiff = (counter & 0xffff) - (counters[serial] & 0xffff);
if(cdiff <= 0) return 5;
if(cdiff > 16) //needs resync
        {
#ifdef debug
        printP(PSTR("resync requested\n"));
#endif
        resync[serial]= counter;
        return 6;
        }
}
else
{
int32_t cdiff = counter - counters[serial];
if(cdiff <= 0) return 5;
if(cdiff > 16) //needs resync
        {
#ifdef debug
        printP(PSTR("resync requested\n"));
#endif
        resync[serial]=counter;
        return 6;
        }
}

counters[serial]=counter; //store recent value
return 0; //success
}

#endif
//COUNTER32


uint32_t makehash(uint32_t password, const uint64_t *salt)
{
//use keeloq as a hash function, in a non-(simply)-invertible way i.e. password as a part of key, not the data to be encrypted 
//this is analogous to usage of DES in /etc/passwd
uint64_t tmp = password;
tmp |= *salt & 0xffffffff00000000LL;
uint32_t seed = *salt&0x00000000ffffffffLL;
encrypt(&seed,&tmp);
return seed;
}

int16_t checkpassword(uint32_t payload, uint8_t npass, const uint32_t *hashes,  uint64_t salt, uint8_t *lockunlock, char **suffix)
{
static char pwbuf[16];
uint8_t i;
*lockunlock = (payload&0x80000000L)?1:0;
payload&= 0x7fffffffL;
itoa10_32(payload,pwbuf);
if(suffix) *suffix=pwbuf+PASSWORD_LENGTH;
char pwbuf2[PASSWORD_LENGTH+1];
strncpy(pwbuf2,pwbuf,PASSWORD_LENGTH+1);
pwbuf2[PASSWORD_LENGTH]=0;
payload=makehash(atoi10_32(pwbuf2),&salt);
//check with respect to all nonzero hashes (zero ones are invalidated passwords)
for(i=0; i<npass; ++i) if(hashes[i]!=0 && payload == hashes[i]) return i;

return -1;
}



#endif
